import Mathlib
section MatrixRank
open Matrix Function LinearMap

/-
 - general theorems:
 -   theorem 1 = max_rank_iff_det_ne_zero
 -   theorem 2 = rank_submatrix_le2
 -/

variable (nF : Type) [Fintype nF] [DecidableEq nF]
variable (K : Type) [Field K]

-- Matrix A is invertible in Mathlib: isUnit A
-- basic theorems about matrix rank in Data.Matrix.Rank

lemma max_rank_if_det_ne_zero (A : Matrix nF nF K) : A.det ≠ 0 → A.rank = Fintype.card nF := by
  intro h
  apply rank_of_isUnit A
  rw [isUnit_iff_isUnit_det]
  simp
  tauto

omit [DecidableEq nF] in
lemma surjective_if_max_rank (A : Matrix nF nF K) :
    A.rank = Fintype.card nF → Surjective A.mulVecLin := by
  intro h1
  apply range_eq_top.mp
  rw [@range_mulVecLin]
  apply Submodule.eq_top_of_finrank_eq
  rw [← @eRank_toNat_eq_finrank]
  rw [Module.finrank_pi]
  rw [eRank_toNat_eq_rank]
  exact h1

lemma det_ne_zero_if_max_rank (A : Matrix nF nF K) : (A.rank = Fintype.card nF) → (A.det ≠ 0) := by
  intro h1
  rw [← isUnit_iff_ne_zero]
  rw [← isUnit_iff_isUnit_det]
  apply mulVec_injective_iff_isUnit.mp
  have h2 : Surjective A.mulVecLin → Injective A.mulVecLin := by
    apply injective_iff_surjective.mpr
  apply h2
  apply surjective_if_max_rank
  exact h1

theorem max_rank_iff_det_ne_zero (A : Matrix nF nF K) :
    (A.det ≠ 0) ↔ (A.rank = Fintype.card nF) := by
  constructor
  · apply max_rank_if_det_ne_zero
  · apply det_ne_zero_if_max_rank

-- a slightly stronger version of Mathlib.Data.Matrix.Rank.rank_submatrix_le,
-- has been included in Mathlib
variable (m : Type) [Fintype m]
variable (m₀ : Type) [Fintype m₀]
variable (n : Type)
variable (n₀ : Type)

theorem rank_submatrix_le (f : n₀ → n) (e : m₀ ≃ m) (A : Matrix n m K) :
    rank (A.submatrix f e) ≤ rank A := by
  rw [Matrix.rank]     -- replace matrix rank by range of linear map (lhs)
  rw [Matrix.rank]     -- replace matrix rank by range of linear map (rhs)
  rw [mulVecLin_submatrix]    -- submatrix-φ = f ∘ₗ matrix-φ ∘ₗ e
  rw [LinearMap.range_comp]   -- range f ∘ₗ φ = f (range φ)
  rw [LinearMap.range_comp]   -- range φ ∘ₗ e = φ (range e)
  have h : LinearMap.funLeft K K e.symm = LinearEquiv.funCongrLeft K K e.symm := by
    rfl  -- e is special: an equivalence
  rw [h]
  rw [LinearEquiv.range]   -- image of e is ⊤
  rw [Submodule.map_top]   -- φ ⊤ = range φ
  exact Submodule.finrank_map_le _ _

/-
 - exercise 3
 -/

def M (x : ℝ) : Matrix (Fin 3) (Fin 3) ℝ := !![1, 2, 3; 1, x, 3; 1, 2, x]

variable (c : ℝ)

-- step 1

lemma det_M : (M c).det = (c - 2)*(c - 3) :=
  calc
    (M c).det = 1*c*c - 1*3*2 - 2*1*c + 2*3*1 + 3*1*2 - 3*c*1 := by
      apply det_fin_three
    _ = (c - 2)*(c - 3) := by
      ring

-- step 2

lemma det_M_eq_zero : (c ≠ 2 ∧ c ≠ 3) ↔ (M c).det ≠ 0 := by
  rw [det_M]
  simp
  apply and_congr
  · exact Iff.symm sub_ne_zero
  · exact Iff.symm sub_ne_zero

lemma rank3_M : (c ≠ 2 ∧ c ≠ 3) ↔ (M c).rank = 3 := by
  rw [det_M_eq_zero]
  rw [max_rank_iff_det_ne_zero]
  simp

-- step 3
def S2 : Matrix (Fin 2) (Fin 3) ℝ := !![1, 2, 3; 1, 2, 2]
def S3 : Matrix (Fin 2) (Fin 3) ℝ := !![1, 2, 3; 1, 3, 3]

open Fin.CommRing
def f2 (i : Fin 2) : (Fin 3) := i + 1
def f3 (i : Fin 2) : (Fin 3) := i
def e : (Fin 3) ≃ (Fin 3) := Equiv.refl (Fin 3)

lemma S2_is_submatrix : S2 = submatrix (M 2) f2 e := by
  exact ext_of_single_vecMul (congrFun rfl)

lemma S3_is_submatrix : S3 = submatrix (M 3) f3 e := by
  exact ext_of_single_vecMul (congrFun rfl)

lemma S2_rank_2 : S2.rank = 2 := by
  rw [← rank_self_mul_transpose]
  apply max_rank_if_det_ne_zero
  have hSt : S2ᵀ = !![1, 1; 2, 2; 3, 2] := by
    exact (etaExpand_eq S2ᵀ).symm
  have h: S2 * S2ᵀ = !![14, 11; 11, 9] := by
    rw [hSt, S2]
    simp
    ring_nf
  rw [h]
  simp
  norm_num

lemma S3_rank_2 : S3.rank = 2 := by
  rw [← rank_self_mul_transpose]
  apply max_rank_if_det_ne_zero
  have hSt : S3ᵀ = !![1, 1; 2, 3; 3, 3] := by
    exact (etaExpand_eq S3ᵀ).symm
  have h: S3 * S3ᵀ = !![14, 16; 16, 19] := by
    rw [hSt, S3]
    simp
    ring_nf
  rw [h]
  simp
  norm_num

-- step 4
lemma rank_ge_2_M : (c = 2 ∨ c = 3) → (M c).rank ≥ 2 := by
  intro h1
  rcases h1 with ha | ha
  · rw [ha]
    have h : S2.rank ≤ (M 2).rank := by
      rw [S2_is_submatrix]
      exact (rank_submatrix_le ℝ (Fin 3) (Fin 3) (Fin 3) (Fin 2) f2 e (M 2))
    rw [S2_rank_2] at h
    exact h
  · rw [ha]
    have h : S3.rank ≤ (M 3).rank := by
      rw [S3_is_submatrix]
      exact (rank_submatrix_le ℝ (Fin 3) (Fin 3) (Fin 3) (Fin 2) f3 e (M 3))
    rw [S3_rank_2] at h
    exact h

lemma rank2_M : (c = 2 ∨ c = 3) → (M c).rank = 2 := by
  intro h1
  have h2 : (M c).det = 0 := by
    by_contra hP
    have h2a: (M c).det ≠ 0 := by
      exact hP
    rw [← det_M_eq_zero] at h2a
    tauto
  have h3 : (M c).rank ≤ 3 := by
    exact rank_le_width (M c)
  have h4 : (M c).rank ≠ 3 := by
    by_contra h41
    apply det_ne_zero_if_max_rank (Fin 3) ℝ
    rw [h41]
    rw [Fintype.card_fin]
    exact h2
  have h5 : (M c).rank ≤ 2 := by
    apply Nat.le_of_lt_succ
    exact Nat.lt_of_le_of_ne h3 h4
  have h6 : (M c).rank ≥ 2 := by
    apply rank_ge_2_M
    exact h1
  exact Eq.symm (Nat.le_antisymm h6 h5)

end MatrixRank
